<?php

namespace Modules\Essentials\Http\Controllers;

use App\Category;
use App\User;
use App\Utils\ModuleUtil;
use App\Utils\TransactionUtil;
use DB;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Modules\Essentials\Entities\EssentialsAttendance;
use Modules\Essentials\Entities\EssentialsHoliday;
use Modules\Essentials\Entities\EssentialsLeave;
use Modules\Essentials\Entities\EssentialsUserSalesTarget;
use Modules\Essentials\Utils\EssentialsUtil;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * All Utils instance.
     */
    protected $moduleUtil;

    protected $essentialsUtil;

    protected $transactionUtil;

    /**
     * Constructor
     *
     * @param  ModuleUtil  $moduleUtil
     * @return void
     */
    public function __construct(
        ModuleUtil $moduleUtil,
        EssentialsUtil $essentialsUtil,
        TransactionUtil $transactionUtil
    ) {
        $this->moduleUtil = $moduleUtil;
        $this->essentialsUtil = $essentialsUtil;
        $this->transactionUtil = $transactionUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function hrmDashboard()
    {
        $business_id = request()->session()->get('user.business_id');
        $is_admin = $this->moduleUtil->is_admin(auth()->user(), $business_id);
        $user_id = auth()->user()->id;

        // Dapatkan lokasi yang diizinkan untuk pengguna yang login
        $permitted_locations = auth()->user()->permitted_locations();

        // Mulai membangun query untuk mengambil data pengguna (staf)
        $users_query = User::where('business_id', $business_id)->user();

        // Jika pengguna tidak memiliki akses ke 'semua' lokasi, terapkan filter
        if ($permitted_locations != 'all') {
            $users_query->whereHas('locations', function ($query) use ($permitted_locations) {
                $query->whereIn('business_locations.id', $permitted_locations);
            });
        }

        // Eksekusi query untuk mendapatkan daftar staf
        $users = $users_query->get();
        
        $departments = Category::where('business_id', $business_id)
            ->where('category_type', 'hrm_department')
            ->get();
        $users_by_dept = $users->groupBy('essentials_department_id');

        $today = new \Carbon('today');
        $one_month_from_today = \Carbon::now()->addMonth();
        
        // Filter data cuti berdasarkan staf yang sudah difilter per lokasi
        $staff_ids = $users->pluck('id');
        
        $leaves = EssentialsLeave::where('business_id', $business_id)
            ->whereIn('user_id', $staff_ids)
            ->where('status', 'approved')
            ->whereDate('end_date', '>=', $today->format('Y-m-d'))
            ->whereDate('start_date', '<=', $one_month_from_today->format('Y-m-d'))
            ->with(['user', 'leave_type'])
            ->orderBy('start_date', 'asc')
            ->get();

        $todays_leaves = [];
        $upcoming_leaves = [];
        $users_leaves = [];
        foreach ($leaves as $leave) {
            $leave_start = \Carbon::parse($leave->start_date);
            $leave_end = \Carbon::parse($leave->end_date);

            if ($today->gte($leave_start) && $today->lte($leave_end)) {
                $todays_leaves[] = $leave;
                if ($leave->user_id == $user_id) {
                    $users_leaves[] = $leave;
                }
            } elseif ($today->lt($leave_start) && $leave_start->lte($one_month_from_today)) {
                $upcoming_leaves[] = $leave;
                if ($leave->user_id == $user_id) {
                    $users_leaves[] = $leave;
                }
            }
        }

        $holidays_query = EssentialsHoliday::where('essentials_holidays.business_id', $business_id)
            ->whereDate('end_date', '>=', $today->format('Y-m-d'))
            ->whereDate('start_date', '<=', $one_month_from_today->format('Y-m-d'))
            ->orderBy('start_date', 'asc')
            ->with(['location']);

        if ($permitted_locations != 'all') {
            $holidays_query->where(function ($query) use ($permitted_locations) {
                $query->whereIn('essentials_holidays.location_id', $permitted_locations)
                      ->orWhereNull('essentials_holidays.location_id');
            });
        }
        $holidays = $holidays_query->get();

        $todays_holidays = [];
        $upcoming_holidays = [];
        foreach ($holidays as $holiday) {
            $holiday_start = \Carbon::parse($holiday->start_date);
            $holiday_end = \Carbon::parse($holiday->end_date);
            if ($today->gte($holiday_start) && $today->lte($holiday_end)) {
                $todays_holidays[] = $holiday;
            } elseif ($today->lt($holiday_start) && $holiday_start->lte($one_month_from_today)) {
                $upcoming_holidays[] = $holiday;
            }
        }

        $todays_attendances = [];
        if ($is_admin) {
            // Filter data absensi berdasarkan staf yang sudah difilter per lokasi
            $todays_attendances = EssentialsAttendance::where('business_id', $business_id)
                ->whereIn('user_id', $staff_ids)
                ->whereDate('clock_in_time', \Carbon::now()->format('Y-m-d'))
                ->with(['employee'])
                ->orderBy('clock_in_time', 'asc')
                ->get();
        }

        $settings = $this->essentialsUtil->getEssentialsSettings();
        $sales_targets = EssentialsUserSalesTarget::where('user_id', $user_id)->get();
        $start_date = \Carbon::today()->startOfMonth()->format('Y-m-d');
        $end_date = \Carbon::today()->endOfMonth()->format('Y-m-d');
        $sale_totals = $this->transactionUtil->getUserTotalSales($business_id, $user_id, $start_date, $end_date);
        $target_achieved_this_month = !empty($settings['calculate_sales_target_commission_without_tax']) && $settings['calculate_sales_target_commission_without_tax'] == 1 ? $sale_totals['total_sales_without_tax'] : $sale_totals['total_sales'];
        $start_date = \Carbon::parse('first day of last month')->format('Y-m-d');
        $end_date = \Carbon::parse('last day of last month')->format('Y-m-d');
        $sale_totals = $this->transactionUtil->getUserTotalSales($business_id, $user_id, $start_date, $end_date);
        $target_achieved_last_month = !empty($settings['calculate_sales_target_commission_without_tax']) && $settings['calculate_sales_target_commission_without_tax'] == 1 ? $sale_totals['total_sales_without_tax'] : $sale_totals['total_sales'];
        $now = \Carbon::now()->addDays(1)->format('Y-m-d');
        $thirtyDaysFromNow = \Carbon::now()->addDays(30)->format('Y-m-d');
        
        // Filter data ulang tahun berdasarkan staf yang sudah difilter per lokasi
        $up_comming_births = User::whereIn('id', $staff_ids)
            ->where('business_id', $business_id)
            ->whereRaw("DATE_FORMAT(dob, '%m-%d') BETWEEN DATE_FORMAT('$now', '%m-%d') AND DATE_FORMAT('$thirtyDaysFromNow', '%m-%d')")
            ->orderBy('dob', 'asc')->get();
        
        $today_births = User::whereIn('id', $staff_ids)
            ->whereMonth('dob', \Carbon::now()->format('m'))
            ->whereDay('dob', \Carbon::now()->format('d'))
            ->get();

        return view('essentials::dashboard.hrm_dashboard')
            ->with(compact('users', 'departments', 'users_by_dept', 'todays_holidays', 'todays_leaves', 'upcoming_leaves', 'is_admin', 'users_leaves', 'upcoming_holidays', 'todays_attendances', 'sales_targets', 'target_achieved_this_month', 'target_achieved_last_month', 'up_comming_births', 'today_births'));
    }

    public function getUserSalesTargets()
    {
        $business_id = request()->session()->get('user.business_id');

        $this_month_start_date = \Carbon::today()->startOfMonth()->format('Y-m-d');
        $this_month_end_date = \Carbon::today()->endOfMonth()->format('Y-m-d');
        $last_month_start_date = \Carbon::parse('first day of last month')->format('Y-m-d');
        $last_month_end_date = \Carbon::parse('last day of last month')->format('Y-m-d');
        $settings = $this->essentialsUtil->getEssentialsSettings();

        $query = User::where('users.business_id', $business_id)
            ->join('transactions as t', 't.res_waiter_id', '=', 'users.id')
            ->where('t.type', 'sell')
            ->whereDate('transaction_date', '>=', $last_month_start_date)
            ->where('t.status', 'final');

        // Filter target penjualan berdasarkan lokasi
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $query->whereIn('t.location_id', $permitted_locations);
        }
        
        if (!empty($settings['calculate_sales_target_commission_without_tax']) && $settings['calculate_sales_target_commission_without_tax'] == 1) {
            $query->select(
                DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name"),
                DB::raw("SUM(IF(DATE(transaction_date) BETWEEN '{$last_month_start_date}' AND '{$last_month_end_date}', total_before_tax - shipping_charges - (SELECT SUM(item_tax*quantity) FROM transaction_sell_lines as tsl WHERE tsl.transaction_id=t.id), 0)) as total_sales_last_month"),
                DB::raw("SUM(IF(DATE(transaction_date) BETWEEN '{$this_month_start_date}' AND '{$this_month_end_date}', total_before_tax - shipping_charges - (SELECT SUM(item_tax*quantity) FROM transaction_sell_lines as tsl WHERE tsl.transaction_id=t.id), 0)) as total_sales_this_month")
            );
        } else {
            $query->select(
                DB::raw("CONCAT(COALESCE(surname, ''), ' ', COALESCE(first_name, ''), ' ', COALESCE(last_name, '')) as full_name"),
                DB::raw("SUM(IF(DATE(transaction_date) BETWEEN '{$last_month_start_date}' AND '{$last_month_end_date}', final_total, 0)) as total_sales_last_month"),
                DB::raw("SUM(IF(DATE(transaction_date) BETWEEN '{$this_month_start_date}' AND '{$this_month_end_date}', final_total, 0)) as total_sales_this_month")
            );
        }

        $query->groupBy('users.id');

        return Datatables::of($query)
            ->editColumn('total_sales_this_month', fn ($row) => $this->transactionUtil->num_f($row->total_sales_this_month, true))
            ->editColumn('total_sales_last_month', fn ($row) => $this->transactionUtil->num_f($row->total_sales_last_month, true))
            ->make(false);
    }

    /**
     * Retrieves employee work hours for DataTables.
     *
     * @return \Illuminate\Http\Response
     */
    public function getEmployeeWorkHours(Request $request)
    {
        $business_id = request()->session()->get('user.business_id');

        // --- MODIFIKASI DIMULAI: Mendapatkan rentang tanggal dari request ---
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        // Default ke "Bulan Ini" jika tidak ada tanggal yang diberikan
        if (empty($start_date)) {
            $start_date = Carbon::now()->startOfMonth()->format('Y-m-d');
        }
        if (empty($end_date)) {
            $end_date = Carbon::now()->endOfMonth()->format('Y-m-d');
        }
        // --- AKHIR MODIFIKASI ---

        // Subquery untuk periode yang dipilih
        $period_hours = DB::table('essentials_attendances as a')
            ->where('a.business_id', $business_id)
            ->whereBetween(DB::raw('DATE(a.clock_in_time)'), [$start_date, $end_date]) // Menggunakan DATE()
            ->whereNotNull('a.clock_out_time')
            ->select(
                'a.user_id',
                DB::raw('SUM(TIMESTAMPDIFF(SECOND, a.clock_in_time, a.clock_out_time)) as total_seconds_in_period'),
                DB::raw('COUNT(DISTINCT DATE(a.clock_in_time)) as attendance_count_in_period')
            )
            ->groupBy('a.user_id');

        $users_query = User::where('users.business_id', $business_id)
            ->where('users.user_type', 'user')
            ->where('users.status', 'active');
        
        $permitted_locations = auth()->user()->permitted_locations();
        if ($permitted_locations != 'all') {
            $users_query->whereHas('locations', function ($query) use ($permitted_locations) {
                $query->whereIn('business_locations.id', $permitted_locations);
            });
        }
        
        $users = $users_query->leftJoinSub($period_hours, 'period_hours', function ($join) {
                $join->on('users.id', '=', 'period_hours.user_id');
            })
            ->select(
                DB::raw("CONCAT(COALESCE(surname, ''), ' ', first_name, ' ', COALESCE(last_name, '')) as employee_name"),
                'period_hours.total_seconds_in_period',
                'period_hours.attendance_count_in_period',
                DB::raw('CASE WHEN period_hours.attendance_count_in_period > 0 THEN (period_hours.total_seconds_in_period / period_hours.attendance_count_in_period) ELSE 0 END as avg_seconds_per_day')
            );

        // Hitung hari kerja pada periode yang dipilih (Senin-Sabtu)
        $start_date_carbon = Carbon::parse($start_date);
        $end_date_carbon = Carbon::parse($end_date);
        $working_days_in_period = 0;
        for ($date = $start_date_carbon; $date->lte($end_date_carbon); $date->addDay()) {
            if (!$date->isSunday()) {
                $working_days_in_period++;
            }
        }
        
        return Datatables::of($users)
            ->addColumn('work_hours_in_period', function ($row) {
                $total_seconds = $row->total_seconds_in_period ?? 0;
                $hours = floor($total_seconds / 3600);
                return $hours . ' Jam';
            })
            ->addColumn('attendance_in_period', function ($row) {
                return ($row->attendance_count_in_period ?? 0) . ' Hari';
            })
            ->addColumn('absence_in_period', function ($row) use ($working_days_in_period) {
                $absence = $working_days_in_period - ($row->attendance_count_in_period ?? 0);
                return $absence . ' Hari';
            })
            ->editColumn('avg_seconds_per_day', function ($row) {
                $avg_seconds_per_day = $row->avg_seconds_per_day ?? 0;
                $hours = floor($avg_seconds_per_day / 3600);
                $minutes = floor(($avg_seconds_per_day % 3600) / 60);

                return sprintf('%d Jam %d Menit', $hours, $minutes);
            })
            ->rawColumns(['attendance_in_period', 'absence_in_period'])
            ->make(true);
    }

    
    public function essentialsDashboard()
    {
        return view('essentials::dashboard.essentials_dashboard');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return view('essentials::create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     * @return Response
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Show the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function show($id)
    {
        return view('essentials::show');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function edit($id)
    {
        return view('essentials::edit');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Request  $request
     * @param  int  $id
     * @return Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy($id)
    {
        //
    }
}